
/*
  CLASSiC DAC, Copyright 2013 SILICON CHIP Publications
  Config.c: functions to load configuration file off SD card and set up global config variables
  Written by Nicholas Vinen, 2012-2013
*/

#include <ctype.h>
#include <string.h>
#include "Config.h"
#include "Playback.h"

int strncasecmp(const char* s1, const char* s2, unsigned int n) {
	while(n) {
		if( *s1 != *s2 )
			return (int)*s1 - (int)*s2;
		if( !*s1 )
			return 0;
		++s1;
		++s2;
		--n;
	}
	return 0;
}

static unsigned char this_token_is(char* buf, unsigned short* buf_pos, unsigned short* buf_len, const char* token) {
	unsigned short toklen = strlen(token);
	if( *buf_len-*buf_pos < toklen || strncasecmp(buf+*buf_pos, token, toklen) )
		return 0;
	*buf_pos += toklen;
	return 1;
}
static unsigned char skip_space(char* buf, unsigned short* buf_pos, unsigned short* buf_len) {
	if( *buf_pos == *buf_len || !isspace(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && isspace(buf[*buf_pos]) && buf[*buf_pos] != '\n' )
		++*buf_pos;
	return 1;
}
static unsigned char next_token_is(char* buf, unsigned short* buf_pos, unsigned short* buf_len, const char* token) {
	unsigned short pos;
	if( !isspace(buf[*buf_pos]) )
		return 0;
	pos = *buf_pos;
	skip_space(buf, &pos, buf_len);
	if( this_token_is(buf, &pos, buf_len, token) ) {
		*buf_pos = pos;
		return 1;
	} else {
		return 0;
    }   
}

static unsigned char get_next_token_boolean(char* buf, unsigned short* buf_pos, unsigned short* buf_len) {
	if( next_token_is(buf, buf_pos, buf_len, "yes") || next_token_is(buf, buf_pos, buf_len, "true") )
		return 1;
	else if( next_token_is(buf, buf_pos, buf_len, "no") || next_token_is(buf, buf_pos, buf_len, "false") )
		return 0;
	else
		return 0xFF;
}

static unsigned char read_integer(char* buf, unsigned short* buf_pos, unsigned short* buf_len, unsigned short* to_here) {
	unsigned short ret = 0;
	if( *buf_len == *buf_pos || !isdigit(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && isdigit(buf[*buf_pos]) ) {
		if( ret > 6553 || (ret == 6553 && buf[*buf_pos] > '5') )
			return 0;
		ret = ret*10 + (buf[*buf_pos]-'0');
		++*buf_pos;
	}
	*to_here = ret;
	return 1;
}
static unsigned char read_signed_integer(char* buf, unsigned short* buf_pos, unsigned short* buf_len, signed short* to_here) {
	signed short ret = 0, neg = 1;
    if( buf[*buf_pos] == '-' ) {
        neg = -1;
		++*buf_pos;
    }
	if( *buf_len == *buf_pos || !isdigit(buf[*buf_pos]) )
		return 0;
	while( *buf_pos < *buf_len && isdigit(buf[*buf_pos]) ) {
		if( ret > 3276 || (ret == 3276 && buf[*buf_pos] > (neg ? '8' : '7')) )
			return 0;
		ret = ret*10 + (buf[*buf_pos]-'0');
		++*buf_pos;
	}
	*to_here = ret * neg;
	return 1;
}

extern unsigned char g_DIPSwitchStates;
void reset_config_to_default(DACConfig* pConfig) {
  memset(pConfig, 0, sizeof(*pConfig));
  pConfig->PlaybackOrder = sorted;

  pConfig->Auto_Play = 1;
  pConfig->Auto_Switch_Enabled = (g_DIPSwitchStates>>1)&1;
  pConfig->AutoSwitchDelay = 10;
  pConfig->AlternativeIRCodes = (g_DIPSwitchStates>>2)&1;

  pConfig->Tone_BassCrossoverFreq = 500;
  pConfig->Tone_TrebleCrossoverFreq = 2000;

  pConfig->Crossfeed_LPFFreq = 1500;
  pConfig->Crossfeed_Delay = 14;
  pConfig->Crossfeed_Atten = 2;
}

static inline unsigned short abs(signed short val) {
	return val >= 0 ? val : -val;
}

unsigned char read_config(FIL* file, DACConfig* pConfig, char* InitFileToHere, int InitFileMaxLen) {
  char buf[256];
  unsigned char ret = 1;
  unsigned short buf_pos = 0, buf_len = 0, temp;
  signed short itemp;
  UINT read;

  while(1) {
    if( f_read(file, (BYTE*)buf + buf_len, sizeof(buf)-buf_len, &read) != FR_OK )
	  return 0;
    buf_len += read;
    if( buf_len == 0 )
      break;
	skip_space(buf, &buf_pos, &buf_len);
    if( buf_pos < buf_len ) {
	  if( buf[buf_pos] == '\n' ) {
        ++buf_pos;
      } else {
		if( this_token_is(buf, &buf_pos, &buf_len, "#") || this_token_is(buf, &buf_pos, &buf_len, "//") ) {
			// skip the comment
			while( buf_pos < buf_len && buf[buf_pos] != '\n' )
				++buf_pos;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Input") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) || temp < 1 || temp > 8 )
				return 0;
			pConfig->Input = temp-1;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Mute") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Mute = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Volume") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) || temp > 255 )
				return 0;
			pConfig->Volume = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Balance") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_signed_integer(buf, &buf_pos, &buf_len, &itemp) || itemp < -20 || itemp > 20 )
				return 0;
			pConfig->Balance = itemp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "PlaybackOrder") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			if( next_token_is(buf, &buf_pos, &buf_len, "directory") )
				pConfig->PlaybackOrder = directory;
			else if( next_token_is(buf, &buf_pos, &buf_len, "sorted") )
				pConfig->PlaybackOrder = sorted;
			else if( next_token_is(buf, &buf_pos, &buf_len, "shuffle") )
				pConfig->PlaybackOrder = shuffle;
			else
				return 0;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Auto_Play") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Auto_Play = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Auto_Switch_Enabled") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Auto_Switch_Enabled = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "AutoSwitchDelay") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) || temp > 999 )
				return 0;
			pConfig->AutoSwitchDelay = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "AlternativeIRCodes") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->AlternativeIRCodes = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Tone_Enabled") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Tone_Enabled = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Tone_EqualVolume") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Tone_EqualVolume = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Tone_AutoLoudness") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Tone_AutoLoudness = temp;
            if( temp ) {
			    pConfig->Tone_BassBoostCut = 0;
			    pConfig->Tone_TrebleBoostCut = 0;
			    pConfig->Tone_BassCrossoverFreq = 500;
			    pConfig->Tone_TrebleCrossoverFreq = 2000;
            }
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Tone_BassBoostCut") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_signed_integer(buf, &buf_pos, &buf_len, &itemp) || itemp < -16 || itemp > 16 )
				return 0;
			pConfig->Tone_BassBoostCut = itemp;
			pConfig->Tone_AutoLoudness = false;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Tone_TrebleBoostCut") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_signed_integer(buf, &buf_pos, &buf_len, &itemp) || itemp < -16 || itemp > 16 )
				return 0;
			pConfig->Tone_TrebleBoostCut = itemp;
			pConfig->Tone_AutoLoudness = false;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Tone_BassCrossoverFreq") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) || temp < 50 || temp > 950 )
				return 0;
			pConfig->Tone_BassCrossoverFreq = temp;
			pConfig->Tone_AutoLoudness = false;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Tone_TrebleCrossoverFreq") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) || temp < 1000 || temp > 5000 )
				return 0;
			pConfig->Tone_TrebleCrossoverFreq = temp;
			pConfig->Tone_AutoLoudness = false;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Crossfeed_Enabled") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Crossfeed_Enabled = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Crossfeed_LPFFreq") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) || temp < 50 || temp > 5000 )
				return 0;
			pConfig->Crossfeed_LPFFreq = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Crossfeed_Delay") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) || temp < 1 || temp > 32 )
				return 0;
			pConfig->Crossfeed_Delay = temp-1;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Crossfeed_Atten") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
			if( !read_integer(buf, &buf_pos, &buf_len, &temp) || temp < 1 || temp > 5 )
				return 0;
			pConfig->Crossfeed_Atten = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "StereoSwap") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->StereoSwap = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "DownmixToMono") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->DownmixToMono = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Filter_Slowrolloff") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Filter_Slowrolloff = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "InvertPolarity") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->InvertPolarity = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "NoFreeRunningPLL") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->NoFreeRunningPLL = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "NoDeEmphasis") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->NoDeEmphasis = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Crossfeed_IgnoreHPSocket") ) {
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
			temp = get_next_token_boolean(buf, &buf_pos, &buf_len);
			if( temp > 1 )
				return 0;
			pConfig->Crossfeed_IgnoreHPSocket = temp;
        } else if( this_token_is(buf, &buf_pos, &buf_len, "Init_File") ) {
            char* dest = InitFileToHere;
            int left = InitFileMaxLen;
			if( !next_token_is(buf, &buf_pos, &buf_len, "=") )
				return 0;
            skip_space(buf, &buf_pos, &buf_len);
            while( buf_pos < buf_len && buf[buf_pos] != '\n' && --left )
              *dest++ = buf[buf_pos++];
            while( dest > InitFileToHere && isspace(dest[-1]) )
              --dest;
            *dest = '\0';
        } else {
          return 0;
        }
	    if( !next_token_is(buf, &buf_pos, &buf_len, "\n") )
           return 0;
      }
    }
    memmove(buf, buf+buf_pos, buf_len-buf_pos);
    buf_len -= buf_pos;
    buf_pos = 0;
  }

  return ret;
}

unsigned char find_and_read_config(DACConfig* pConfig, char* InitFileToHere, int InitFileMaxLen) {
	FIL file;
	if( f_open(&file, "0:\\" CONFIG_FILENAME, FA_READ|FA_OPEN_EXISTING) == FR_OK ) {
		return read_config(&file, pConfig, InitFileToHere, InitFileMaxLen);
	} else {
		return 1;
	}
}
